import { useState, useEffect, useRef } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, Star, Package, CheckCircle, Award, Apple, TrendingUp, BarChart3, Zap, Target, Users, Clock } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyConsultations.css'

const MyConsultations = () => {
  const navigate = useNavigate()
  const [consultations, setConsultations] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('nutriConsultations')
    if (saved) {
      try {
        setConsultations(JSON.parse(saved))
      } catch (e) {
        setConsultations([])
      }
    } else {
      const defaultConsultations = [
        {
          id: 1,
          plan: 'Сбалансированное питание',
          date: '2025-01-20',
          rating: 5
        },
        {
          id: 2,
          plan: 'Детокс программа',
          date: '2025-01-10',
          rating: 4
        },
        {
          id: 3,
          plan: 'План для набора массы',
          date: '2024-12-28',
          rating: 5
        }
      ]
      setConsultations(defaultConsultations)
      localStorage.setItem('nutriConsultations', JSON.stringify(defaultConsultations))
    }
  }, [])

  const formatDate = (dateString) => {
    if (!dateString) return 'Не указано'
    try {
      const [year, month, day] = dateString.split('-').map(Number)
      if (!year || !month || !day) return 'Не указано'
      
      const date = new Date(year, month - 1, day)
      if (isNaN(date.getTime())) return 'Не указано'
      
      return date.toLocaleDateString('ru-RU', { 
        day: 'numeric', 
        month: 'long', 
        year: 'numeric' 
      })
    } catch (e) {
      return 'Не указано'
    }
  }

  return (
    <div className="my-consultations-page-wellness">
      <div className="container">
        <section className="page-header-wellness">
          <div className="header-badge-wellness">
            <Apple size={18} />
            <span>ВАШИ КОНСУЛЬТАЦИИ</span>
          </div>
          <h1 className="page-title-wellness">
            Мои <span className="title-accent-wellness">консультации</span>
          </h1>
          <p className="page-description-wellness">
            Отслеживайте свои планы питания и результаты консультаций
          </p>
        </section>

        <ImageSlider />

        <GlassmorphismSection />

        <StatsSection />

        {consultations.length === 0 ? (
          <div className="empty-state-wellness">
            <div className="empty-icon-wrapper-wellness">
              <Package size={64} />
            </div>
            <h2 className="empty-title-wellness">Нет консультаций</h2>
            <p className="empty-text-wellness">Запишитесь на первую консультацию с нутрициологом</p>
            <button 
              className="empty-button-wellness" 
              onClick={() => navigate('/enroll')}
            >
              Записаться на консультацию
            </button>
          </div>
        ) : (
          <section className="consultations-list-wellness">
            <div className="consultations-section-header-wellness">
              <h2 className="consultations-section-title-wellness">Мои записи на консультации</h2>
              <p className="consultations-section-description-wellness">
                Все ваши планы питания и результаты работы с нутрициологом
              </p>
            </div>
            <div className="consultations-grid-wellness">
              {consultations.map((consultation) => (
                <ConsultationCard
                  key={consultation.id}
                  consultation={consultation}
                  formatDate={formatDate}
                />
              ))}
            </div>
          </section>
        )}
      </div>
    </div>
  )
}

const ConsultationCard = ({ consultation, formatDate }) => {
  return (
    <div className="consultation-item-wellness">
      <div className="consultation-header-wellness">
        <div className="consultation-icon-wrapper-wellness">
          <Apple size={28} />
        </div>
        <div className="consultation-info-wellness">
          <div className="consultation-number-wellness">#{consultation.id}</div>
          <div className="rating-badge-wellness">
            <Star size={14} fill="#FDCB6E" color="#FDCB6E" />
            <span>{consultation.rating}</span>
          </div>
        </div>
      </div>
      
      <h3 className="consultation-title-wellness">{consultation.plan}</h3>
      
      <div className="consultation-details-wellness">
        <div className="detail-item-wellness">
          <Calendar size={16} />
          <div>
            <span className="detail-label-wellness">Дата консультации:</span>
            <span className="detail-value-wellness">{formatDate(consultation.date)}</span>
          </div>
        </div>
        <div className="detail-item-wellness">
          <Star size={16} />
          <div>
            <span className="detail-label-wellness">Оценка результатов:</span>
            <div className="rating-stars-wellness">
              {Array.from({ length: 5 }).map((_, i) => (
                <Star
                  key={i}
                  size={14}
                  fill={i < consultation.rating ? "#FDCB6E" : "none"}
                  color="#FDCB6E"
                />
              ))}
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}

const GlassmorphismSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const features = [
    {
      icon: Zap,
      title: 'Быстрый результат',
      value: '2 недели',
      description: 'Первые положительные изменения',
      gradient: 'linear-gradient(135deg, #6C5CE7 0%, #A29BFE 100%)'
    },
    {
      icon: Target,
      title: 'Точность плана',
      value: '98%',
      description: 'Индивидуальный подход к каждому',
      gradient: 'linear-gradient(135deg, #00B894 0%, #55EFC4 100%)'
    },
    {
      icon: Award,
      title: 'Экспертиза',
      value: '10+ лет',
      description: 'Опыт работы наших специалистов',
      gradient: 'linear-gradient(135deg, #FD79A8 0%, #FDCB6E 100%)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`glassmorphism-section-wellness ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="glassmorphism-header-wellness">
          <h2 className="glassmorphism-title-wellness">Наши преимущества</h2>
          <p className="glassmorphism-description-wellness">
            Почему выбирают NutriWell
          </p>
        </div>
        <div className="glassmorphism-grid-wellness">
          {features.map((feature, index) => {
            const Icon = feature.icon
            return (
              <div 
                key={index} 
                className="glassmorphism-card-wellness"
                style={{ 
                  animationDelay: `${index * 0.15}s`,
                  background: feature.gradient
                }}
              >
                <div className="glassmorphism-content-wellness">
                  <div className="glassmorphism-icon-wrapper-wellness">
                    <Icon size={32} />
                  </div>
                  <div className="glassmorphism-value-wellness">{feature.value}</div>
                  <h3 className="glassmorphism-title-card-wellness">{feature.title}</h3>
                  <p className="glassmorphism-text-wellness">{feature.description}</p>
                </div>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

const StatsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const stats = [
    {
      icon: Users,
      title: '5000+ клиентов',
      description: 'Доверяют нам свое здоровье',
      color: 'var(--wellness-purple)'
    },
    {
      icon: Target,
      title: '95% успеха',
      description: 'Достигают поставленных целей',
      color: 'var(--wellness-cyan)'
    },
    {
      icon: Zap,
      title: 'Быстрый результат',
      description: 'Первые изменения за 2 недели',
      color: 'var(--wellness-pink)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`stats-section-wellness ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="stats-header-wellness">
          <h2 className="stats-title-wellness">Наши достижения</h2>
          <p className="stats-description-wellness">
            Цифры, которые говорят сами за себя
          </p>
        </div>
        <div className="stats-grid-wellness">
          {stats.map((stat, index) => {
            const Icon = stat.icon
            return (
              <div 
                key={index} 
                className="stat-card-wellness"
                style={{ 
                  animationDelay: `${index * 0.15}s`,
                  '--stat-color': stat.color
                }}
              >
                <div className="stat-icon-wrapper-wellness">
                  <Icon size={36} />
                </div>
                <div className="stat-value-wellness">{stat.title}</div>
                <p className="stat-text-wellness">{stat.description}</p>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

export default MyConsultations
